var ui = {
  // Define device types, layout (tabs), and fields for each
  // id corresponds to FIRMWARE_ID (APP_FW_ID) 
  deviceTypes: [
      /*{"id":24, "alternateIds":[7], "name":"echoUAT", "type":"transceiver-978", defaultControl:5, setupPacketVersion: 4, solicitStatusReport: true},
      {"id":29, "name":"skyEcho", "type":"transceiver-1090", defaultControl:3, setupPacketVersion: 4, solicitStatusReport: false},
      {"id":33, "name":"skyBeacon", "type":"transceiver-978", defaultControl:5, setupPacketVersion: 4, solicitStatusReport: true},
      {"id":28, "name":"echoESX", "type":"transponder", defaultControl:58, setupPacketVersion: 5, solicitStatusReport: false},
      {"id":26, "name":"Vektor", "type":"transceiver-1090", defaultControl:3, setupPacketVersion: 4, solicitStatusReport: false},*/
      {"id":25, "alternateIds":[55], "name":"echo VTU-20", "type":"transmitter-978", defaultControl:5, setupPacketVersion: 4, solicitStatusReport: false}
      /*{"id":9, "name":"ATT-20B", "type":"transceiver-1090", defaultControl:3, setupPacketVersion: 4, solicitStatusReport: false} */
  ],
  SETUP: {APP: 0x00,
          EFIS: 0x01},
  CONTROL: {NONE: 0x00,
            EFIS: 0x01,
            SNIFFER: 0x02,
            APP: 0x04,
            NOT_PROVIDED: 0xFF},
  GPS: {NONE: 0x00,
        EFIS: 0x01,
        EXTGPS: 0x02,
        NOT_PROVIDED: 0xFF},
  DISPLAY: {NONE: 0x00,
            APP: 0x01,
            EFIS: 0x02,
            MFD: 0x04,
            NOT_PROVIDED: 0xFF},
  PROTOCOL: {MAVLINK: 0x0001,
             GDL90: 0x0002,
             DYNON: 0x0004,
             TMAP: 0x0008,
             SAGETECH: 0x0010,
             GTX: 0x0020,
             GDL: 0x0040,
             NMEA: 0x0080,
             ADSB_PLUS: 0x0100,
             NOT_PROVIDED: 0xFFFF},
  defaultDeviceIndex: 0,
  currentDevice: null,
  layouts: null,
  fields: null,

  // Add persistance if desired
  latLonFormat: "dms",
  altitudeUnits: "ft",
  
  remote: require('electron').remote,
  
  comPortTimeout: null,

  // UI Constructor
  initialize: function() {
    // Layouts are currently unused - a future concept. We dynamically add Monitor currently.
    this.layouts = new Object();
    this.layouts["ping2020"] = ["Status"];
    this.layouts["ping1090"] = ["Status"];
    this.layouts["ping200S"] = ["Status", "Monitor"];

    this.fields = new Object();
    this.fields["ALL"] = ["transceiverConfiguration", "transponderConfiguration", "installation", "icao", "emitter", "callsign", "vLengthLabel", "aLengthLabel", "aLengthInput", "vWidthLabel", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "vfrCode", "maxSpeed", "adsbIn", "sda", "sil", "threshold", "controlLabel", "defaultControlLabel", "control", "outputFormat", "flightPlanId", "csidLogic", "anonymous", "setupSource", "controlSources", "gpsSource", "displayOutput", "com1Rate", "com1Data", "com1PHY", "com1Protocol", "com2Rate", "com2Data", "com2PHY", "com2InputProtocol", "com2OutputProtocol", "baroAltSource", "geofence", "geofenceFile", "update", "updateAdvanced", "skyBeaconMessage"];
    this.fields["echoUAT"] = ["transceiverConfiguration", "installation", "icao", "emitter", "callsign", "aLengthLabel", "aLengthInput", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "vfrCode", "adsbIn", "sda", "sil", "threshold", "controlLabel", "control", "flightPlanId", "csidLogic", "anonymous", "setupSource", "controlSources", "gpsSource", "displayOutput", "com1Rate", "com1Data", "com1PHY", "com1Protocol", "com2Rate", "com2InputProtocol", "update", "updateAdvanced"];
    this.fields["skyEcho"] = ["transceiverConfiguration", "icao", "emitter", "callsign", "aLengthLabel", "aLengthInput", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "sda", "sil", "controlLabel", "control", "update", "updateAdvanced"];
    this.fields["skyBeacon"] = ["skyBeaconMessage"];
    //this.fields["skyBeacon"] = ["transceiverConfiguration", "icao", "emitter", "callsign", "aLengthLabel", "aLengthInput", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "adsbIn", "sda", "sil", "threshold", "controlLabel", "control", "anonymous", "update", "updateAdvanced"];
    //this.fields["ATT-20B"] = ["transceiverConfiguration", "icao", "emitter", "callsign", "aLengthLabel", "aLengthInput", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "sda", "sil", "controlLabel", "control", "update", "updateAdvanced"];
    this.fields["echoESX"] = ["transponderConfiguration", "installation", "icao", "emitter", "callsign", "aLengthLabel", "aLengthInput", "aWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "Vs0", "vfrCode", "adsbIn", "sda", "sil", "defaultControlLabel", "control", "flightPlanId", "setupSource", "gpsSource", "com1PHY", "com2Rate", "com2InputProtocol", "baroAltSource", "update", "updateAdvanced"];
    this.fields["Vektor"] = ["transceiverConfiguration", "icao", "emitter", "callsign", "vLengthLabel", "aLengthInput", "vWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "sda", "sil", "controlLabel", "control", "geofence", "geofenceFile", "update", "updateAdvanced"];
    this.fields["echo VTU-20"] = ["transmitterConfiguration", "icao", "emitter", "callsign", "vLengthLabel", "aLengthInput", "vWidthLabel", "aWidthInput", "offsetLat", "offsetLon", "sda", "sil", "controlLabel", "control", "geofence", "geofenceFile", "update", "updateAdvanced"];

    // Fields that either have data-device-id-exclude or data-device-type set
    // and require processing to limit options
    this.maskedFields = ["control", "com1PHY", "emitter"];

    // Mandatory field support, only partially implemented
    // Place required fields for regular and advanced update here
    this.mandatoryFields = new Object;
    this.mandatoryFields["echoUAT"] = ["vfrCode", "Vs0", "sda", "sil", "threshold"];
    //this.mandatoryFields["ATT-20B"] = ["icao", "callsign", "Vs0", "sda", "sil"];
    this.mandatoryFields["skyEcho"] = ["icao", "callsign", "Vs0", "sda", "sil"];
    this.mandatoryFields["skyBeacon"] = ["icao", "callsign", "vfrCode", "Vs0", "sda", "sil", "threshold"];
    this.mandatoryFields["echoESX"] = ["icao", "callsign", "vfrCode", "Vs0", "sda", "sil"];
    this.mandatoryFields["Vektor"] = ["icao", "callsign", "sda", "sil"];
    this.mandatoryFields["echo VTU-20"] = ["icao", "callsign", "sda", "sil"];

    this.populateWidths();

    this.populateComPorts();
    this.comPortTimeout = setInterval(this.populateComPorts.bind(this), 1000);
    this.populateDeviceTypes();
    this.deviceTypeUpdated("");
    console.log("UI intialized");
  },
  isMandatory: function(device, field) {
    var index;
    var mandatoryFieldsLength = this.mandatoryFields[device.name].length;
    for (index = 0; index < mandatoryFieldsLength; index++) {
      if (this.mandatoryFields[device.name][index] == field) {
        return true;
      }
    }
    return false;
  },
  updateFieldDisplay: function(display, fieldNames) {
    var index;
    var fieldsLength = this.fields[fieldNames].length;
    var fieldName;
    for (index = 0; index < fieldsLength; index++) {
      // Display or hide the span which includes the label, not just the field itself
      fieldName = this.fields[fieldNames][index] + "Span";
      document.getElementById(fieldName).style.display = display;
    }
  },
  // Compare primary ID and alternate IDs to find device table entry
  deviceIdToDevice: function(deviceId) {
    var index;
    var deviceTypesLength = this.deviceTypes.length;
    for (index = 0; index < deviceTypesLength; index++) {
      if (this.deviceTypes[index].id == deviceId) {
        return this.deviceTypes[index];
      }
      var altId = this.deviceTypes[index].alternateIds;
      if (Array.isArray(altId)) {
        for (var idIndex = 0; idIndex < altId.length; idIndex++ ) {
          if (altId[idIndex] == deviceId) {
            return this.deviceTypes[index];
          }
        }
      } else {
        if (altId == deviceId) {
          return this.deviceTypes[index];
        }
      }
    }
    return null;
  },
  // controlValue is what we want the control value to be set to
  // It may be device specific
  deviceTypeUpdated: function(controlValue) {
    // Set up display for the selected device
    // Clear display
    this.updateFieldDisplay("none", "ALL");

    // Find device
    var deviceId = document.getElementById("deviceType").value;
    this.currentDevice = this.deviceIdToDevice(deviceId);

    var index;
    for (index = 0; index < this.maskedFields.length; index++) {
      var fieldToMask = document.getElementById(this.maskedFields[index]);
      var currentValue = fieldToMask.value;

      // Hide options based on device type
      // There is no cross-platform way to hide select options, so we need to
      // cache originals and work from there
      if (fieldToMask.innerHTMLOriginal == undefined) {
        // First time, cache full options
        fieldToMask.innerHTMLOriginal = fieldToMask.innerHTML;
      } else {
        // Subsequent, restore and remove unecessary items
        fieldToMask.innerHTML = fieldToMask.innerHTMLOriginal;
      }

      // Restore current value
      fieldToMask.value = currentValue;

      // Mask first by data-device-type (dataset.deviceType)
      if (fieldToMask.dataset.maskType == "deviceType") {
        for (var i=fieldToMask.options.length-1; i>=0; i--) {
          if ((fieldToMask.item(i).dataset.deviceType != "all") && 
              (fieldToMask.item(i).dataset.deviceType != this.currentDevice.type)) {
            fieldToMask.remove(i);
          }
        }
      }
      // Mask second by data-device-id-exclude (dataset.deviceIdExclude)
      // Excludes can be arrays or single entries
      if (fieldToMask.dataset.maskType == "deviceIdExclude") {
        for (var i=fieldToMask.options.length-1; i>=0; i--) {
          if (fieldToMask.item(i).dataset.deviceIdExclude != undefined) {
            var exclude = JSON.parse(fieldToMask.item(i).dataset.deviceIdExclude);
            if (Array.isArray(exclude)) {
              for (var excludeIndex = 0; excludeIndex < exclude.length; excludeIndex++ ) {
                if (exclude[excludeIndex] == this.currentDevice.id) {
                  fieldToMask.remove(i);
                }
              }
            } else {
              if (exclude == this.currentDevice.id) {
                fieldToMask.remove(i);
              }
            }
          }
        }
      }

    }

    // Try to re-select cached / appropriate control options
    // Default to device specific value if not found
    var controlSelect = document.getElementById("control");
    controlSelect.value = controlValue;
    if (controlSelect.selectedIndex == -1) {
      controlSelect.value = this.currentDevice.defaultControl;
    }
    
    this.updateFieldDisplay("block", this.currentDevice.name);
  },
  validateICAO: function(icaoValue) {
    // Match against our list of invalid ICAOs
    var INVALID_ICAOS = ['000000', 'FFFFFF'];
    for (var i=0; i < INVALID_ICAOS.length; i++) {
      if (icaoValue.toUpperCase() == INVALID_ICAOS[i]) {
        return false;
      }
    }

    var re = /^[0-9A-Fa-f]{6}$/g;
    if (re.test(icaoValue)) {
      return true;
    } else {
      return false;
    }
  },
  validateCallsign: function(callsignValue) {
    var re = /^[0-9A-Z]{1,8}$/g;
    if (re.test(callsignValue)) {
      return true;
    } else {
      return false;
    }
  },
  validateLengthWidth: function(aLength, aWidth) {
    if (isNaN(aLength) || aWidth.charAt(0) != aLength) {
      return false;
    } else {
      return true;
    }
  },
  validateGPSOffset: function(offsetLat, offsetLon) {
    if (isNaN(offsetLat) || isNaN(offsetLon)) {
      return false;
    } else {
      return true;
    }
  },
  validateVs0: function(vs0Value) {
    var re = /^[0-9]{1,3}$/g;
    if (re.test(vs0Value)) {
      return true;
    } else {
      return false;
    }
  },
  validateSda: function(sdaValue) {
    var re = /^[0-9]{1,1}$/g;
    if (re.test(sdaValue) && sdaValue <= 3) {
      return true;
    } else {
      return false;
    }
  },
  validateSil: function(silValue) {
    var re = /^[0-9]{1,1}$/g;
    if (re.test(silValue) && silValue <= 3) {
      return true;
    } else {
      return false;
    }
  },
  validateThreshold: function(thresholdValue) {
    var re = /^[0-9]{1,5}$/g;
    if (re.test(thresholdValue) && thresholdValue <= 65534) {
      return true;
    } else {
      return false;
    }
  },
  validateDefaultVfrCode: function(vfrCodeValue) {
    var re = /^[0-7]{1,4}$/g;
    if (re.test(vfrCodeValue) && vfrCodeValue <= 7777) {
      return true;
    } else {
      return false;
    }
  },
  // RCB TODO validate both 4 and 6 digit flight plan IDs
  validateFlightPlanId: function(flightPlanIdValue) {
    var re = /^[0-7]{1,4}$/g;
    if (re.test(flightPlanIdValue) && flightPlanIdValue <= 7777) {
      return true;
    } else {
      return false;
    }
  },
  validateComPorts: function() {
    // Look for conflicting port assignments
    // Setup, Control and GPS can share COM1 if TMAP
    // Traffic must be the sole function of COM1
    // COM2 can be split between GPS and Traffic (different input/output protocols)
    if (document.getElementById("displayOutput").value & this.DISPLAY.EFIS &&
       (document.getElementById("setupSource").value == this.SETUP.EFIS ||
        document.getElementById("controlSources").value & this.CONTROL.EFIS ||
        document.getElementById("gpsSource").value == this.GPS.EFIS)) {
      return false;
    } else {
      return true;
    }
  },
  validateCom1Protocol: function() {
    if (document.getElementById("displayOutput").value & this.DISPLAY.EFIS) {
      // We need a traffic protocol
      if (document.getElementById("com1Protocol").value != this.PROTOCOL.GDL90 &&
          document.getElementById("com1Protocol").value != this.PROTOCOL.DYNON) {
        return false;
      }
    }
    if (document.getElementById("gpsSource").value == this.GPS.EFIS) {
      // We need TMAP
      if (document.getElementById("com1Protocol").value != this.PROTOCOL.TMAP) {
        return false;
      }
    }
    if (document.getElementById("setupSource").value == this.SETUP.EFIS) {
      // We need TMAP
      if (document.getElementById("com1Protocol").value != this.PROTOCOL.TMAP) {
        return false;
      }
    }
    if (document.getElementById("controlSources").value & this.CONTROL.EFIS) {
      // Only allow TMAP, GTX or GDL Control
      if (document.getElementById("com1Protocol").value != this.PROTOCOL.TMAP &&
          document.getElementById("com1Protocol").value != this.PROTOCOL.GTX &&
          document.getElementById("com1Protocol").value != this.PROTOCOL.GDL) {
        return false;
      }
    }
    return true;
  },
  formatICAO: function() {
    document.getElementById("icao").value = document.getElementById("icao").value.toUpperCase();
  },
  formatCallsign: function() {
    document.getElementById("callsign").value = document.getElementById("callsign").value.toUpperCase();
  },
  populateComPorts: function() {
    serial.getPorts();
  },
  populateComPortsCallback: function(portNames) {
    var comPortSelect = document.getElementById("comPort");
    var currentSelection = comPortSelect.value;
    for (var i = comPortSelect.options.length - 1 ; i >= 0 ; i--) {
        comPortSelect.remove(i);
    }
    portNames.forEach(function(portName) {
      var option = document.createElement("option");
      option.value = portName;
      option.text = portName;
      comPortSelect.add(option);
    });
    if (currentSelection != "") { 
      comPortSelect.value = currentSelection;
    }
    if (currentSelection != comPortSelect.value) {
      app.connectComPort();
    }
  },
  populateDeviceTypes: function() {
    var deviceTypeSelect = document.getElementById("deviceType");
    var i;
    // Clear existing devices
    for (i=deviceTypeSelect.options.length-1; i>=0; i--) {
      deviceTypeSelect.remove(i);
    }

    for (i=this.deviceTypes.length-1; i>=0; i--) {
      var option = document.createElement("option");
      option.value = this.deviceTypes[i].id;
      option.text = this.deviceTypes[i].name;
      deviceTypeSelect.add(option, deviceTypeSelect[0]);
    }

    // Select default item
    deviceTypeSelect.selectedIndex = this.defaultDeviceIndex;
  },
  populateWidths: function() {
    <!-- Note conflicting tables between DO-282B and the ICAO UAT manual (derived from an earlier DO-282? -->
    <!-- We will use DO-282B -->
    var widths = [
      {"value":"0.1", "text":"W \u2264 23"},
      {"value":"1.0", "text":"W \u2264 28.5"},
      {"value":"1.1", "text":"28.5 < W \u2264 34"},
      {"value":"2.0", "text":"W \u2264 33"},
      {"value":"2.1", "text":"33 < W \u2264 38"},
      {"value":"3.0", "text":"W \u2264 39.5"},
      {"value":"3.1", "text":"39.5 < W \u2264 45"},
      {"value":"4.0", "text":"W \u2264 45"},
      {"value":"4.1", "text":"45 < W \u2264 52"},
      {"value":"5.0", "text":"W \u2264 59.5"},
      {"value":"5.1", "text":"59.5 < W \u2264 67"},
      {"value":"6.0", "text":"W \u2264 72.5"},
      {"value":"6.1", "text":"72.5 < W \u2264 80"},
      {"value":"7.0", "text":"W \u2264 80"},
      {"value":"7.1", "text":"W > 80"},
      {"value":"8.1", "text":"Any"}
    ];

    var aLength = document.getElementById("aLength").value;
    var widthSelect = document.getElementById("aWidth");

    var i;
    // Clear existing options
    for (i=widthSelect.options.length-1; i>=0; i--) {
      widthSelect.remove(i);
    }

    // Add matching per length
    for (i=widths.length-1; i>=0; i--) {
      if (widths[i].value.charAt(0) == aLength) {
        var option = document.createElement("option");
        option.value = widths[i].value;
        option.text = widths[i].text;
        widthSelect.add(option, widthSelect[0]);
      }
    }

    // Select first item
    widthSelect.selectedIndex = 0;
  },
  // Validate data
  // Valid if all the fields are valid
  // Optional fields need to be selectively sent in upper level logic
  validateData: function() {
    var dataValid = true;
    var errorMessage = "";
    if ((this.isMandatory(this.currentDevice, "icao") || document.getElementById("icao").value.trim().length != 0) &&
        !(this.validateICAO(document.getElementById("icao").value))) {
      dataValid = false;
      errorMessage += "ICAO";
    }
    if ((this.isMandatory(this.currentDevice, "callsign") || document.getElementById("callsign").value.trim().length != 0) &&
        !(this.validateCallsign(document.getElementById("callsign").value))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", callsign";
      } else {
        errorMessage += "callsign";
      }
    }
    if (!(this.validateLengthWidth(document.getElementById("aLength").value, document.getElementById("aWidth").value))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", dimensions";
      } else {
        errorMessage += "dimensions";
      }
    }
    if (!(this.validateGPSOffset(document.getElementById("offsetLat").value, document.getElementById("offsetLon").value))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", GPS offset";
      } else {
        errorMessage += "GPS offset";
      }
    }
                                                                          
    // Validate device specific (not always present) data
    // If a field is visible assume we will be sending it
    if (document.getElementById("Vs0Span").style.display != "none" &&
       ((this.isMandatory(this.currentDevice, "Vs0") || document.getElementById("Vs0").value.trim().length != 0) &&
        !(this.validateVs0(document.getElementById("Vs0").value)))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", Vs0";
      } else {
        errorMessage += "Vs0";
      }
    }

    if (document.getElementById("vfrCodeSpan").style.display != "none" &&
       ((this.isMandatory(this.currentDevice, "vfrCode") || document.getElementById("vfrCode").value.trim().length != 0) &&
        !(this.validateDefaultVfrCode(document.getElementById("vfrCode").value)))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", VFR Code";
      } else {
        errorMessage += "VFR Code";
      }
    }

    if (document.getElementById("flightPlanIdSpan").style.display != "none" &&
       ((this.isMandatory(this.currentDevice, "flightPlanId") || document.getElementById("flightPlanId").value.trim().length != 0) &&
        !(this.validateDefaultVfrCode(document.getElementById("flightPlanId").value)))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", Flight Plan ID";
      } else {
        errorMessage += "Flight Plan ID";
      }
    }

    if ((document.getElementById("setupSourceSpan").style.display != "none" ||
        document.getElementById("controlSourcesSpan").style.display != "none" ||
        document.getElementById("gpsSourceSpan").style.display != "none" ||
        document.getElementById("displayOutputSpan").style.display != "none")) {
      if (!(this.validateComPorts())) {
        dataValid = false;
        if (errorMessage.length) {
          errorMessage += ", COM Port Configuration (COM1 cannot output Traffic while configured for another function)";
        } else {
          errorMessage += "COM Port Configuration (COM1 cannot output Traffic while configured for another function)";
        }
      } else {
        // Check protocols
        if (document.getElementById("com1ProtocolSpan").style.display != "none" &&
            !(this.validateCom1Protocol())) {
          dataValid = false;
          if (errorMessage.length) {
            errorMessage += ", COM1 protocol (choose a valid protocol for selected functions)";
          } else {
            errorMessage += "COM1 protocol (choose a valid protocol for selected functions)";
          }
        }
      }
    }

    if (!dataValid) {
      errorMessage = "Invalid " + errorMessage;
	    //this.remote.dialog.showMessageBox({message: errorMessage, title: "Error", buttons: ["OK"]});
	    this.remote.dialog.showErrorBox("Invalid Parameters", errorMessage);
      //navigator.notification.alert(errorMessage, null, "Error", "OK");
    }
                                                                          
    return dataValid;
  },
  // Validate advanced data
  // Valid if all the fields are valid
  // Optional fields need to be selectively sent in upper level logic
  validateAdvanced: function() {
    var dataValid = true;
    var errorMessage = "";
    if ((this.isMandatory(this.currentDevice, "sda") || document.getElementById("sda").value.trim().length != 0) &&
        !(this.validateSda(document.getElementById("sda").value))) {
      dataValid = false;
      errorMessage += "SDA";
    }
    if ((this.isMandatory(this.currentDevice, "sil") || document.getElementById("sil").value.trim().length != 0) &&
        !(this.validateSil(document.getElementById("sil").value))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", SIL";
      } else {
        errorMessage += "SIL";
      }
    }
    if ((this.isMandatory(this.currentDevice, "threshold") || document.getElementById("threshold").value.trim().length != 0) &&
        !(this.validateThreshold(document.getElementById("threshold").value))) {
      dataValid = false;
      if (errorMessage.length) {
        errorMessage += ", threshold";
      } else {
        errorMessage += "threshold";
      }
    }

    if (!dataValid) {
      errorMessage = "Invalid " + errorMessage;
      //navigator.notification.alert(errorMessage, null, "Error", "OK");
	    this.remote.dialog.showErrorBox("Invalid Parameters", errorMessage);
    }

    return dataValid;
  },
  // Toggle lat/lon display format from DMS to decimal degrees
  toggleLatLonDisplayFormat: function() {
    if (this.latLonFormat === "dms") {
      this.latLonFormat = "dd";
    } else {
      this.latLonFormat = "dms";
    }
    app.updateMonitorInfo();
  },
  // Toggle altitude display units from ft to meters
  toggleAltitudeDisplayUnits: function() {
    if (this.altitudeUnits === "ft") {
      this.altitudeUnits = "m";
    } else {
      this.altitudeUnits = "ft";
    }
    app.updateMonitorInfo();
  }
};
